# Django
from django.db.models import Count, Sum, Q
from django.db.models.functions import Coalesce

# Django Rest Framework
from rest_framework.permissions import IsAdminUser
from rest_framework.viewsets import GenericViewSet
from rest_framework.mixins import (
    ListModelMixin,
    RetrieveModelMixin,
    CreateModelMixin,
    DestroyModelMixin
)

# Project
from ..serializers.car import CarAdminSerializer
from ...models import Car
from ..filters import CarAdminFilterSet


class CarAdminViewSet(ListModelMixin, RetrieveModelMixin, CreateModelMixin,
                      DestroyModelMixin, GenericViewSet):
    permission_classes = (IsAdminUser,)
    serializer_class = CarAdminSerializer
    filterset_class = CarAdminFilterSet

    def get_queryset(self):
        return Car.raw_objects.select_related(
            'user', 'car_brand', 'car_model', 'car_detail').annotate(
            fines_remain=Coalesce(Sum(
                'car_detail__fines__remain',
                filter=Q(car_detail__fines__pStatus__in=[1, 3])
            ), 0),
            fines_count=Count('car_detail__fines', filter=Q(
                car_detail__fines__pStatus__in=[1, 3],
            ))
        )
