# Django
from django.conf import settings
from django.db.models import Sum, FloatField
from django.db.models.functions import Coalesce, Cast
from django.contrib.postgres.fields.jsonb import KeyTextTransform
from django.utils import timezone
from django.template.loader import render_to_string

# Pypi: celery
from celery import shared_task
from datetime import timedelta

# Project
from .bot import bot
from apps.gauth.models import User
from apps.cars.models import Fine
from apps.transactions.models import Check


def models_count(model, **kwargs):
    return model.objects.filter(
        created_date__gte=timezone.now() - timedelta(days=1), **kwargs).count()


@shared_task(bind=True)
def daily_reports(self):
    data = {
        'date': timezone.now(),
        'users': models_count(User),
        'offences': models_count(Fine, pStatus=1),
        'offences_closed': models_count(Fine, pStatus=5),
        'offences_paid': models_count(Fine, pStatus=4),
        'offences_paid_sum': int(Fine.objects.filter(
            created_date__gte=timezone.now() - timedelta(days=1),
            pStatus=4).aggregate(sum=Coalesce(Sum('pAmount'), 0))['sum']),
        'offences_paid_road24': models_count(Check),
        'offences_paid_road24sum': int(Check.objects.filter(
            created_date__gte=timezone.now() - timedelta(days=1)).annotate(
            to_int=Cast(KeyTextTransform(
                'amount', KeyTextTransform('payer', 'result')),
                FloatField())).aggregate(sum=Coalesce(Sum('to_int'), 0))[
                                           'sum']) // 100,
        'users_with_no_cars': models_count(User, cars=None),
        'clock': u'\U0001f550',
        'man': u'\U0001f468',
        'money': u'\U0001f4b0',
        'dollar': u'\U0001f4b2',
        'paid': u'\U0001f4b5',
        'sum': u'\U0001f4b2',
        'pedestrian': u'\U0001f6b6'
    }
    resp = render_to_string("format.html", data)
    bot.send_message(settings.TELEGRAM_AUTOPOST_CHANNEL, resp,
                     parse_mode='html')
