# Django
from django.utils.translation import ugettext_lazy as _
from django.utils import timezone

# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from apps.shared.tests import ViewSetTestCase, AuthTestCase
from apps.shared import messages
from ..models import BankCard


class BankCardTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'cards.yaml'
    ]

    url = 'mobile:bank_cards-%s'

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self.client.credentials()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._create({}, {'method': 'resend-sms-code', 'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._create({}, {'method': 'verify', 'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_reqierements_field(self):
        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            "number": [messages.INVALID_REQUIRED],
            "full_name": [messages.INVALID_REQUIRED],
            "expire": [messages.INVALID_REQUIRED],
        })

    def test_retrieve(self):
        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data, {
            "id": 1,
            "number": '8600123412341234',
            "full_name": "Elon Musk",
            "expire": "0101",
            "is_verified": True,
            "filial": ""
        })

    def test_list(self):
        response = self._list()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 2)
        self.assertEqual(len(response.data['results']), 2)
        self.assertEqual(
            response.data['results'][0],
            {
                "id": 2,
                "number": '8600123412341235',
                "full_name": "Elon Musk",
                "expire": "0101",
                "filial": "",
                "is_verified": False
            }
        )

    def test_pagination(self):
        response = self._list({'page_size': 2})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 2)
        self.assertEqual(len(response.data['results']), 2)

    def test_create(self):
        data = {
            "number": '8600123412341230',
            "full_name": "Elon Musk",
            "expire": "0101"
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)
        self.assertEqual(response.data, {
            "id": response.data['id'],
            "number": '8600123412341230',
            "full_name": "Elon Musk",
            "expire": "0101",
            "is_verified": False,
            'filial': ''
        })

        bank_card = BankCard.objects.get(id=response.data['id'])
        self.assertTrue(bank_card.sms_code)

    def test_create_fail_number(self):
        response = self._create({
            "number": '8600123412341235',
            "full_name": "Elon Musk",
            "expire": "0101"
        })
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            "number": [_("Already exists.")]
        })

    def test_constraints(self):
        data = {
            "number": '8600123412341235',
            "full_name": "Elon Musk",
            "expire": "0101"
        }
        BankCard.objects.filter(
            number=data['number'],
            user=self._get_user(self._admin)
        ).update(is_deleted=True)
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)

    def test_update_partial(self):
        data = {
            "full_name": "Hello World!",
        }
        response = self._update({'pk': 1}, data, partial=True)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['full_name'], data['full_name'])

    def test_update_partial_fail_number(self):
        data = {
            "number": '8600123412341235',
        }
        response = self._update({'pk': 1}, data, partial=True)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            "number": [_("Already exists.")]
        })

    def test_update(self):
        data = {
            "id": 1,
            "number": '86001234123233',
            "full_name": "Elon Musk4533",
            "expire": "2222",
            "is_verified": False
        }
        response = self._update({'pk': 1}, data)
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['full_name'], data['full_name'])
        self.assertTrue(response.data['is_verified'])
        self.assertEqual(response.data['number'], '8600123412341234')
        self.assertEqual(response.data['expire'], '0101')

    def test_update_fail_number(self):
        data = {
            "id": 1,
            "number": '8600123412341235',
            "full_name": "Elon Musk4533",
            "expire": "0101"
        }
        response = self._update({'pk': 1}, data)
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {
            "number": [_("Already exists.")]
        })

    def test_update_not_found(self):
        response = self._update({'pk': 3})
        self.assertEqual(response.status_code, status.HTTP_404_NOT_FOUND)

    def test_destroy(self):
        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_204_NO_CONTENT)
        bank_card = BankCard.raw_objects.get(id=1)
        self.assertEqual(bank_card.is_deleted, True)

    def test_verify(self):
        bank_card = BankCard.objects.get(id=2)
        self.assertFalse(bank_card.is_verified)
        bank_card.sms_date = timezone.now()
        bank_card.save()

        response = self._create(
            {'code': '123123'},
            {'method': 'verify', 'pk': 2}
        )
        bank_card.refresh_from_db()
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertTrue(bank_card.is_verified)
        self.assertEqual(bank_card.sms_code, '')

    def test_verify_fail_already(self):
        response = self._create(
            {'code': '123123'},
            {'method': 'verify', 'pk': 1}
        )
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {'code': [_('Already verified.')]})

    def test_verify_fail_wrong_code(self):
        response = self._create(
            {'code': '3333'},
            {'method': 'verify', 'pk': 2}
        )
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(response.data, {'code': [_('Wrong code.')]})

    def test_resend_fail_already_verified(self):
        response = self._create({}, {'method': 'resend-sms-code', 'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_400_BAD_REQUEST)
        self.assertEqual(
            response.data,
            {'non_field_errors': [_('Already verified.')]}
        )

    def test_resend_success(self):
        self._auth_non_admin()
        now = timezone.now()
        bank_card = BankCard.objects.get(id=3)
        sms_code = bank_card.sms_code
        self.assertEqual(str(bank_card.sms_date.date()), "2017-01-01")
        self.assertTrue(sms_code)

        response = self._create({}, {'method': 'resend-sms-code', 'pk': 3})
        self.assertEqual(response.status_code, status.HTTP_200_OK)

        bank_card.refresh_from_db()
        self.assertEqual(bank_card.sms_date.date(), now.date())
        self.assertNotEqual(bank_card.sms_code, sms_code)
