# Django Rest Framework
from rest_framework.permissions import IsAuthenticated
from rest_framework.viewsets import ModelViewSet
from rest_framework.decorators import action
from rest_framework.response import Response
from rest_framework.throttling import UserRateThrottle
from rest_framework.validators import ValidationError

# Project
from ..models import BankCard
from .serializers import BankCardSerializer, VerifyCardSerializer
from ..services import send_sms_verification


class BankCardViewSet(ModelViewSet):
    permission_classes = (IsAuthenticated,)

    def get_serializer_class(self):
        if self.action == 'verify':
            return VerifyCardSerializer
        return BankCardSerializer

    def get_queryset(self):
        return BankCard.objects.filter(user=self.request.user)

    @action(methods=['POST'], detail=True)
    def verify(self, request, *args, **kwargs):
        instance = self.get_object()
        serializer = self.get_serializer(instance, data=request.data)
        serializer.is_valid(raise_exception=True)
        serializer.save()
        return Response()

    @action(methods=['POST'], detail=True, throttle_classes=[UserRateThrottle])
    def resend_sms_code(self, request, *args, **kwargs):
        instance = self.get_object()
        try:
            send_sms_verification(instance)
        except AssertionError as e:
            raise ValidationError({
                'non_field_errors': [e]
            })
        return Response()
