# Stdlib
from datetime import timedelta

# Django
from django.utils.translation import ugettext as _
from django.conf import settings
from django.db import transaction
from django.utils import timezone

# Django Rest Framework
from rest_framework import serializers

# Project
from ..models import BankCard
from ..utils import get_bank_card_detail
from ..services import send_sms_verification
from apps.shared.serializers import DynamicFieldsModelSerializer


class BankCardSerializer(DynamicFieldsModelSerializer):
    class Meta:
        model = BankCard
        fields = (
            'id',
            'number',
            'full_name',
            'filial',
            'expire',
            'is_verified'
        )
        read_only_fields = ('is_verified', 'filial')

    def validate_number(self, number):
        user = self.context['request'].user
        if BankCard.objects.filter(number=number, user=user).exists():
            raise serializers.ValidationError(_("Already exists."))

        return number

    def validate(self, attrs):
        request = self.context['request']
        language = request.LANGUAGE_CODE

        if not self.instance:
            number = attrs['number']
            expire = attrs['expire']

            if settings.ENV == 'production':
                is_success, data = get_bank_card_detail(number, expire)
                if not is_success:
                    lang_error = 'ru' if language == 'ru' else 'uz'
                    raise serializers.ValidationError(
                        data[lang_error]
                    )
                self.detail_card = data
        return attrs

    def update(self, instance, validated_data):
        validated_data.pop('number', None)
        validated_data.pop('expire', None)
        return super().update(instance, validated_data)

    @transaction.atomic
    def create(self, validated_data):
        validated_data['user'] = self.context['request'].user

        if settings.ENV == 'production':
            validated_data['phone'] = self.detail_card['phone']
            validated_data['filial'] = self.detail_card['filial']

        instance = super().create(validated_data)
        send_sms_verification(instance)
        return instance


class VerifyCardSerializer(serializers.Serializer):
    code = serializers.CharField()

    def validate_code(self, code):
        now = timezone.now()
        instance: BankCard = self.instance

        if instance.is_verified:
            raise serializers.ValidationError(_('Already verified.'))
        if code != instance.sms_code:
            raise serializers.ValidationError(_('Wrong code.'))

        is_expire = instance.sms_date + timedelta(seconds=60) < now
        if is_expire:
            raise serializers.ValidationError(_('Expired.'))

        return code

    def update(self, instance, validated_data):
        instance.sms_code = ''
        instance.sms_date = None
        instance.is_verified = True
        instance.save()
        return validated_data
