# Python
from json import loads, dumps

# Django Rest Framework
from rest_framework.test import APITestCase
from rest_framework import status

# Project
from ..models import Alert
from apps.shared.tests import ViewSetTestCase, AuthTestCase


class ProblemAlertTest(APITestCase, ViewSetTestCase, AuthTestCase):
    fixtures = [
        'user.yaml',
        'region.yaml',
        'district.yaml',
        'file.yaml',
        'address.yaml',
        'problem.yaml',
        'vote.yaml',
        'alerts.yaml',
        'translations.yaml'
    ]

    url = 'myadmin:alerts-%s'

    def setUp(self):
        self._auth_admin()

    def test_unauth(self):
        self.client.credentials()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._list({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_non_admin(self):
        self._auth_non_admin()

        response = self._create({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._list({})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._retrieve({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

        response = self._destroy({'pk': 1})
        self.assertEqual(response.status_code, status.HTTP_403_FORBIDDEN)

    def test_list_uz(self):
        data = {
            "id": 3,
            "title": "uz_lang_title",
            "description": "uz_lang_desc",
            "photo": {
                "id": 3,
                "name": "my_file3",
                "file": "http://testserver/media/test.jpg",
                "video_preview": None,
                "module": ""
            },
            "job_type": "closing",
            "is_published": False,
            "created_date": "2017-01-01T00:00:00.100000+05:00"
        }
        response = self._list(kwargs={
            'headers': {
                'HTTP_ACCEPT_LANGUAGE': 'uz'
            }
        })
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)
        self.assertEqual(len(response.data['results']), 3)
        alert = response.data['results'][0]
        self.assertEqual(loads(dumps(alert)), data)

    def test_list_ru(self):
        data = {
            "id": 3,
            "title": "ru_lang",
            "description": "ru_lang_desc",
            "photo": {
                "id": 3,
                "name": "my_file3",
                "file": "http://testserver/media/test.jpg",
                "video_preview": None,
                "module": ""
            },
            "job_type": "closing",
            "is_published": False,
            "created_date": "2017-01-01T00:00:00.100000+05:00"
        }
        response = self._list(kwargs={
            'headers': {
                'HTTP_ACCEPT_LANGUAGE': 'ru'
            }
        })
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)
        self.assertEqual(len(response.data['results']), 3)
        alert = response.data['results'][0]
        self.assertEqual(loads(dumps(alert)), data)

    def test_pagination(self):
        response = self._list({'page_size': 2})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 3)
        self.assertEqual(len(response.data['results']), 2)

    def test_filter_is_published(self):
        response = self._list({'is_published': True})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 2)
        self.assertEqual(len(response.data['results']), 2)
        self.assertEqual(response.data['results'][0]['id'], 2)

    def test_filter_job_type(self):
        response = self._list({'job_type': Alert.REMOUNT})
        self.assertEqual(response.status_code, status.HTTP_200_OK)
        self.assertEqual(response.data['count'], 1)
        self.assertEqual(len(response.data['results']), 1)
        self.assertEqual(response.data['results'][0]['id'], 2)

    def test_create(self):
        data = {
            "translations": {
                "ru": {
                    "title": "ru_lang_title",
                    "description": "ru_lang_desc"
                },
                "uz": {
                    "title": "uz_lang_title",
                    "description": "uz_lang_desc"
                }
            },
            "photo": 1,
            "job_type": "closing",
            "is_published": True
        }
        response = self._create(data)
        self.assertEqual(response.status_code, status.HTTP_201_CREATED)

        alert = Alert.objects.language('ru').get(id=response.data['id'])
        self.assertEqual(alert.title, data['translations']['ru']['title'])
        self.assertEqual(alert.description,
                         data['translations']['ru']['description'])
        self.assertEqual(alert.photo.id, data['photo'])
        self.assertEqual(alert.job_type, data['job_type'])
        self.assertEqual(alert.is_published, data['is_published'])

        alert = Alert.objects.language('uz').get(id=response.data['id'])
        self.assertEqual(alert.title, data['translations']['uz']['title'])
        self.assertEqual(alert.description,
                         data['translations']['uz']['description'])
        self.assertEqual(alert.photo.id, data['photo'])
        self.assertEqual(alert.job_type, data['job_type'])
        self.assertEqual(alert.is_published, data['is_published'])

    def test_update_partial(self):
        data = {
            'id': 1,
            "translations": {
                "ru": {
                    "title": "ru_lang_title",
                    "description": "ru_lang_desc"
                },
                "uz": {
                    "title": "uz_lang_title",
                    "description": "uz_lang_desc"
                }
            },
            "photo": 1,
            "job_type": "closing",
            "is_published": True
        }
        response = self._update({'pk': 1}, data, partial=True)
        self.assertEqual(response.data['title'],
                         data['translations']['ru']['title'])
        self.assertEqual(response.data['description'],
                         data['translations']['ru']['description'])
        self.assertEqual(response.data['photo']['id'], data['photo'])
        self.assertEqual(response.data['job_type'], data['job_type'])
        self.assertEqual(response.data['is_published'], data['is_published'])

    def test_update(self):
        data = {
            'id': 1,
            "translations": {
                "ru": {
                    "title": "ru_lang_title",
                    "description": "ru_lang_desc"
                },
                "uz": {
                    "title": "uz_lang_title",
                    "description": "uz_lang_desc"
                }
            },
            "photo": 1,
            "job_type": "closing",
            "is_published": True
        }
        response = self._update({'pk': 1}, data)
        self.assertEqual(dict(response.data), data)
